<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * ========================================
 * SiteWay Payment Gateway - النسخة المُصلَّحة
 * ========================================
 * التعديلات:
 * 1. إصلاح الـ API endpoints
 * 2. إضافة معالجة أفضل للأخطاء
 * 3. دعم Estimates بجانب Invoices
 * 4. تحسين الـ logging
 * ========================================
 */

class Sitesway_gateway extends App_gateway
{
    public function __construct()
    {
        parent::__construct();

        $this->setId('sitesway');
        $this->setName('SiteWay');

        $this->setSettings([
            [
                'name'      => 'api_key',
                'encrypted' => false, // Changed from true to false
                'label'     => 'SiteWay API Key',
            ],
            [
                'name'      => 'brand_id',
                'encrypted' => false,
                'label'     => 'SiteWay Brand ID',
            ],
            [
                'name'  => 'test_mode_enabled',
                'type'  => 'yes_no',
                'label' => 'Test Mode',
            ],
            [
                'name'          => 'currencies',
                'label'         => 'Currencies',
                'default_value' => 'SAR',
            ],
        ]);
    }

    /**
     * معالجة الدفع - Invoice Payment
     */
    public function process_payment($data)
    {
        $payment_url = $this->createPayment($data, 'invoice');
        
        if ($payment_url) {
            redirect($payment_url);
        } else {
            set_alert('danger', 'خطأ في إنشاء رابط الدفع. يرجى المحاولة لاحقاً.');
            $invoice = $this->ci->invoices_model->get($data['invoiceid']);
            redirect(site_url('invoice/' . $invoice->id . '/' . $invoice->hash));
        }
    }

    /**
     * إنشاء رابط الدفع - يدعم Invoice و Estimate
     * 
     * @param array $data بيانات الدفع
     * @param string $type نوع الطلب: 'invoice' أو 'estimate'
     * @return string|false رابط الدفع أو false في حالة الخطأ
     */
    private function createPayment($data, $type = 'invoice')
    {
        // جلب الإعدادات
        $api_key  = $this->getSetting('api_key');
        $brand_id = $this->getSetting('brand_id');
        $test     = $this->getSetting('test_mode_enabled') == '1';

        // Fix encrypted API key issue
        if (class_exists('App_encryption') && strpos($api_key, 'enc:') !== false) {
            try {
                $encryption = new App_encryption();
                $api_key = $encryption->decrypt($api_key);
            } catch (Exception $e) {
                log_activity('SiteWay: Failed to decrypt API key - ' . $e->getMessage());
            }
        }

        // التحقق من الإعدادات
        if (empty($api_key) || empty($brand_id)) {
            log_activity('SiteWay Error: API Key or Brand ID missing');
            set_alert('danger', 'خطأ: API Key أو Brand ID غير موجود في الإعدادات');
            return false;
        }

        // Log the API key length for debugging (without exposing the key)
        log_activity('SiteWay: API Key length: ' . strlen($api_key) . ' chars, Brand ID: ' . $brand_id);

        // تحديد الـ API endpoint
        $api_url = $test
            ? 'https://sandbox.gate.sitesway.sa/api/v1/purchases/'
            : 'https://gate.sitesway.sa/api/v1/purchases/';

        // جلب بيانات الفاتورة أو العرض
        if ($type == 'estimate') {
            $this->ci->load->model('estimates_model');
            $record = $this->ci->estimates_model->get($data['id']);
            $record_id = $data['id'];
            $amount = $data['amount'];
            $reference_prefix = 'EST-';
        } else {
            $record = $this->ci->invoices_model->get($data['invoiceid']);
            $record_id = $data['invoiceid'];
            $amount = $data['amount'];
            $reference_prefix = 'INV-';
        }

        if (!$record) {
            log_activity('SiteWay Error: Record not found - Type: ' . $type . ' ID: ' . $record_id);
            return false;
        }

        // جلب بريد العميل
        $email = '';
        if (!empty($record->clientid)) {
            $this->ci->db->where('userid', $record->clientid);
            $this->ci->db->where('is_primary', 1);
            $contact = $this->ci->db->get(db_prefix() . 'contacts')->row();
            if ($contact) {
                $email = $contact->email;
            }
        }

        // تنسيق المبلغ
        $amount = number_format($amount, 2, '.', '');

        // إعداد الـ payload حسب متطلبات SiteWay API الصحيحة
        $purchase = [
            'products' => [
                [
                    'name'     => $type == 'estimate' ? 'Estimate #' . format_estimate_number($record_id) : 'Invoice #' . format_invoice_number($record_id),
                    'price'    => $amount,
                    'quantity' => 1,
                ]
            ],
            'success_redirect' => site_url('gateways/sitesway/success/' . $type . '/' . $record_id),
            'failure_redirect' => site_url('gateways/sitesway/failure/' . $type . '/' . $record_id),
        ];
        
        // Only add webhook if not localhost
        $base_url = site_url();
        if (strpos($base_url, 'localhost') === false && strpos($base_url, '127.0.0.1') === false) {
            $purchase['success_callback'] = site_url('gateways/sitesway/webhook');
        }
        
        // brand_id and client must be at root level
        $payload = [
            'brand_id' => $brand_id,
            'client'   => [
                'email' => $email ?: 'noemail@example.com',
            ],
            'purchase' => $purchase,
        ];

        // إرسال الطلب للـ API
        $ch = curl_init($api_url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_HTTPHEADER     => [
                'Authorization: Bearer ' . $api_key,
                'Content-Type: application/json',
            ],
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_TIMEOUT    => 30,
            CURLOPT_SSL_VERIFYPEER => false, // Disable SSL verification for localhost/test environments
            CURLOPT_SSL_VERIFYHOST => false, // Disable SSL host verification
            CURLOPT_SSLVERSION     => CURL_SSLVERSION_TLSv1_2, // Force TLS 1.2
        ]);

        $response = curl_exec($ch);
        $code     = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);

        // تسجيل النشاط
        log_activity('SiteWay Create Payment Request - Type: ' . $type . ' - Response Code: ' . $code . ' - Response: ' . $response);

        // معالجة الرد
        if ($curl_error) {
            $error_msg = 'خطأ في الاتصال: ' . $curl_error;
            log_activity('SiteWay CURL Error: ' . $curl_error);
            set_alert('danger', $error_msg);
            return false;
        }

        // Log the full request for debugging
        log_activity('SiteWay API Request - URL: ' . $api_url . ' - Brand: ' . $brand_id . ' - Amount: ' . $amount . ' - Currency: ' . ($this->getSetting('currencies') ?: 'SAR'));

        if (in_array($code, [200, 201, 202])) {
            $res = json_decode($response, true);
            
            if (!empty($res['checkout_url'])) {
                // حفظ معرف الدفع في قاعدة البيانات
                if ($type == 'estimate') {
                    // Check if estimate_meta table exists, create if not
                    if (!$this->ci->db->table_exists(db_prefix() . 'estimate_meta')) {
                        $this->ci->db->query("
                            CREATE TABLE IF NOT EXISTS `" . db_prefix() . "estimate_meta` (
                              `meta_id` int(11) NOT NULL AUTO_INCREMENT,
                              `estimate_id` int(11) NOT NULL,
                              `meta_key` varchar(191) NOT NULL,
                              `meta_value` longtext,
                              PRIMARY KEY (`meta_id`),
                              KEY `estimate_id` (`estimate_id`),
                              KEY `meta_key` (`meta_key`)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
                        ");
                    }
                    
                    // Save purchase ID
                    $this->ci->db->replace(db_prefix() . 'estimate_meta', [
                        'estimate_id' => $record_id,
                        'meta_key'    => 'sitesway_purchase_id',
                        'meta_value'  => $res['id'],
                    ]);
                } else {
                    update_invoice_meta($record_id, 'sitesway_purchase_id', $res['id']);
                }
                
                log_activity('SiteWay Success: Payment link created - ' . $res['checkout_url']);
                return $res['checkout_url'];
            } else {
                $error_msg = 'لم يتم إرجاع رابط الدفع من SiteWay. Response: ' . $response;
                log_activity('SiteWay Error: No checkout_url in response - ' . $response);
                set_alert('danger', $error_msg);
            }
        } else {
            // Decode error response
            $error_details = json_decode($response, true);
            
            // Special handling for authentication errors
            if ($code == 401) {
                $error_msg = '❌ خطأ في المصادقة (401): الـ API Key غير صحيح أو منتهي الصلاحية. يرجى التحقق من إعدادات SiteWay.';
                if (!empty($error_details['__all__'][0]['message'])) {
                    $error_msg .= ' تفاصيل: ' . $error_details['__all__'][0]['message'];
                }
            } else {
                $error_msg = 'خطأ من SiteWay API (HTTP ' . $code . ')';
                
                if (!empty($error_details['__all__'][0]['message'])) {
                    $error_msg .= ': ' . $error_details['__all__'][0]['message'];
                } elseif (!empty($error_details['message'])) {
                    $error_msg .= ': ' . $error_details['message'];
                } elseif (!empty($error_details['error'])) {
                    $error_msg .= ': ' . $error_details['error'];
                } else {
                    $error_msg .= ': ' . $response;
                }
            }
            
            log_activity('SiteWay Error: HTTP ' . $code . ' - ' . $response);
            set_alert('danger', $error_msg);
        }

        return false;
    }

    /**
     * صفحة النجاح
     */
    public function success()
    {
        // estimate/123 أو invoice/123
        $type = $this->ci->uri->segment(4); // 'estimate' or 'invoice'
        $id   = $this->ci->uri->segment(5);

        if ($type == 'estimate') {
            $this->ci->load->model('estimates_model');
            $record = $this->ci->estimates_model->get($id);
            $redirect_url = site_url('estimate/' . $id . '/' . $record->hash);
            $message = 'جاري معالجة الدفع، سيتم تأكيد الدفع قريباً.';
        } else {
            $record = $this->ci->invoices_model->get($id);
            $redirect_url = site_url('invoice/' . $id . '/' . $record->hash);
            $message = 'Payment processing, please wait for confirmation.';
        }

        if (!$record) {
            show_404();
        }

        set_alert('success', $message);
        redirect($redirect_url);
    }

    /**
     * صفحة الفشل
     */
    public function failure()
    {
        $type = $this->ci->uri->segment(4);
        $id   = $this->ci->uri->segment(5);

        if ($type == 'estimate') {
            $this->ci->load->model('estimates_model');
            $record = $this->ci->estimates_model->get($id);
            $redirect_url = site_url('estimate/' . $id . '/' . $record->hash);
            $message = 'فشلت عملية الدفع. يرجى المحاولة مرة أخرى.';
        } else {
            $record = $this->ci->invoices_model->get($id);
            $redirect_url = site_url('invoice/' . $id . '/' . $record->hash);
            $message = 'Payment failed. Please try again.';
        }

        if (!$record) {
            show_404();
        }

        set_alert('danger', $message);
        redirect($redirect_url);
    }

    /**
     * Webhook لاستقبال تأكيد الدفع
     */
    public function webhook()
    {
        $payload = file_get_contents('php://input');
        $data = json_decode($payload, true);

        log_activity('SiteWay Webhook Received: ' . $payload);

        // التحقق من البيانات
        if (empty($data['id']) || empty($data['status'])) {
            log_activity('SiteWay Webhook Error: Missing id or status');
            http_response_code(400);
            exit;
        }

        // التحقق من حالة الدفع
        if (!in_array($data['status'], ['paid', 'executed'])) {
            log_activity('SiteWay Webhook: Payment not completed - Status: ' . $data['status']);
            http_response_code(200);
            exit;
        }

        // البحث عن الفاتورة/العرض
        $record_info = $this->findRecordByPurchaseId($data['id']);
        
        if (!$record_info) {
            log_activity('SiteWay Webhook Error: Record not found for purchase_id: ' . $data['id']);
            http_response_code(200);
            exit;
        }

        // تسجيل الدفع للـ Invoice فقط (Estimates لا تُسجل كمدفوعة)
        if ($record_info['type'] == 'invoice') {
            $invoice = $this->ci->invoices_model->get($record_info['id']);
            
            if ($invoice && $invoice->status != 2) { // 2 = Paid
                $this->ci->invoices_model->add_payment([
                    'amount'        => $invoice->total,
                    'invoiceid'     => $record_info['id'],
                    'paymentmode'   => $this->getId(),
                    'transactionid' => $data['id'],
                    'date'          => date('Y-m-d'),
                ]);
                
                log_activity('SiteWay: Payment recorded for Invoice #' . $record_info['id']);
            }
        } else {
            // للـ Estimates، يمكن إضافة نوتة أو تحديث الحالة
            log_activity('SiteWay: Payment confirmed for Estimate #' . $record_info['id']);
        }

        http_response_code(200);
        exit;
    }

    /**
     * البحث عن الفاتورة/العرض بواسطة Purchase ID
     */
    private function findRecordByPurchaseId($purchase_id)
    {
        // البحث في الـ Invoices
        $this->ci->db->where('meta_key', 'sitesway_purchase_id');
        $this->ci->db->where('meta_value', $purchase_id);
        $meta = $this->ci->db->get(db_prefix() . 'invoice_meta')->row();

        if ($meta) {
            return ['type' => 'invoice', 'id' => $meta->invoice_id];
        }

        // البحث في الـ Estimates
        $this->ci->db->where('meta_key', 'sitesway_purchase_id');
        $this->ci->db->where('meta_value', $purchase_id);
        $meta = $this->ci->db->get(db_prefix() . 'estimate_meta')->row();

        if ($meta) {
            return ['type' => 'estimate', 'id' => $meta->estimate_id];
        }

        return null;
    }

    /**
     * دالة مساعدة لإنشاء رابط دفع Estimate من الـ Admin
     * 
     * @param int $estimate_id
     * @param float $amount
     * @return string|false
     */
    public function createEstimatePaymentLink($estimate_id, $amount)
    {
        return $this->createPayment([
            'id'     => $estimate_id,
            'amount' => $amount,
        ], 'estimate');
    }
}